<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\DepositCoin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class DepositController extends Controller
{
    // Show all deposits
    public function index(Request $request)
    {
        $page_title = 'My Deposits';

        $deposits = user()
            ->deposits()
            ->with('depositCoin')
            ->when($request->s, fn($query, $search) => $query->where('ref', 'LIKE', '%' . $search . '%'))
            ->orderBy('id', 'DESC')
            ->paginate(site('pagination'));

        $coins = DepositCoin::where('status', '1')->get();

        return view(template('user.deposits.index'), compact('page_title', 'deposits', 'coins'));
    }

    // Show a single deposit
    public function deposit(Request $request)
    {
        $deposit = user()->deposits()->where('ref', $request->route('ref'))->firstOrFail();

        return response()->json([
            'deposit' => [
                'amount' => $deposit->amount,
                'fee' => $deposit->fee,
                'currency' => $deposit->currency,
                'converted_amount' => $deposit->converted_amount,
                'ref' => $deposit->ref,
                'network' => $deposit->network,
                'valid_until' => $deposit->valid_until,
                'payment_id' => $deposit->payment_id,
                'payment_wallet' => $deposit->payment_wallet,
                'status' => $deposit->status,
            ]
        ]);
    }

    // New deposit
    public function newDeposit(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric',
            'currency_code' => 'required',
        ]);

        $amount_before_fee = $request->amount;
        $currency = $request->currency_code;
        $fee = site('deposit_fee') / 100 * $amount_before_fee;
        $amount = $fee + $amount_before_fee;

        if ($amount_before_fee < site('min_deposit') || $amount_before_fee > site('max_deposit')) {
            return response()->json(validationError('Min or max deposit amount not met'), 422);
        }

        $coin = DepositCoin::where('code', $currency)->where('status', 1)->first();
        if (!$coin) {
            return response()->json(validationError('The Payment method you have selected is not allowed'), 422);
        }

        $processor = site('payment_processor') ?? 'nowpayment';

        $start = initiateDeposit($amount, $currency, $processor);
        if (!$start) {
            return response()->json(validationError('Error initiating deposit'), 422);
        }

        $details = json_decode($start);

        $deposit = Deposit::create([
            'user_id' => user()->id,
            'amount' => $amount_before_fee,
            'fee' => $fee,
            'currency' => $currency,
            'converted_amount' => $details->pay_amount,
            'ref' => $details->order_id,
            'network' => $details->network,
            'valid_until' => $details->valid_until,
            'payment_id' => $details->payment_id,
            'payment_wallet' => $details->pay_address,
            'status' => $details->payment_status,
            'deposit_coin_id' => $coin->id,
        ]);

        return response()->json([
            'deposit' => [
                'amount' => $deposit->amount,
                'fee' => $deposit->fee,
                'currency' => $deposit->currency,
                'converted_amount' => $deposit->converted_amount,
                'ref' => $deposit->ref,
                'network' => $deposit->network,
                'valid_until' => $deposit->valid_until,
                'payment_id' => $deposit->payment_id,
                'payment_wallet' => $deposit->payment_wallet,
                'status' => $deposit->status,
                'link' => $details->checkout_link ?? null,
            ]
        ]);
    }

    // ✅ Initiate Paystack Deposit
    public function initiatePaystackDeposit($amount, $email, $reference)
    {
        $response = Http::withToken(config('services.paystack.secret_key'))->post(
            config('services.paystack.payment_url') . '/transaction/initialize',
            [
                'email' => $email,
                'amount' => $amount * 100, // in kobo
                'reference' => $reference,
                'callback_url' => config('services.paystack.callback_url'),
            ]
        );

        if ($response->successful()) {
            return $response->json()['data']['authorization_url'];
        }

        throw new \Exception('Failed to initiate Paystack payment.');
    }

    // ✅ Callbacks
    public function depositCallback()
    {
        return depositCallback();
    }

    public function depositCallbackCoinpayment()
    {
        return depositCallbackCoinpayment();
    }
}
